﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Collections.Generic;
using System.Linq;
using System.ServiceModel;
using VA.PPMS.Context;

namespace VA.PPMS.CRM.Plugins
{
    public class HistoryLogValidationCreate : IPlugin
    {
        ITracingService _tracingService;
        IOrganizationService _service;
        private const string PluginName = "HistoryLogValidationCreate";

        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            _tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                Log("Begin");

                // Obtain the target entity from the input parameters.
                Entity target = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (target.LogicalName != "ppms_historylog")
                    return;

                Log("Entity found");

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                _service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    OptionSetValue historyLogType = target.GetAttributeValue<OptionSetValue>("ppms_type");

                    //If we dont have a History Log Type, end logic. 
                    if (historyLogType == null)
                        return;

                    Log("History Log Type: {0}", historyLogType.Value);

                    //If this is not a Validation History Log Type we don't want to run any of the logic.
                    if (historyLogType.Value != (int)PpmsHelper.HistoryLogType.Contact &&
                        historyLogType.Value != (int)PpmsHelper.HistoryLogType.Leie &&
                        historyLogType.Value != (int)PpmsHelper.HistoryLogType.NppesNpi &&
                        historyLogType.Value != (int)PpmsHelper.HistoryLogType.Sam &&
                        historyLogType.Value != (int)PpmsHelper.HistoryLogType.CareSite)
                    {
                        Log("Not a Validation History Log");
                        return;
                    }

                    // Get related entities to deactivate
                    Log("Retrieve provider related entities");
                    var providerRef = target.GetAttributeValue<EntityReference>("ppms_providerid");
                    var provider = GetProvider(_service, providerRef.Id);
                    if (provider == null)
                    {
                        Log("Failed to retrieve provider");
                        return;
                    }

                    List<SetStateRequest> requests = new List<SetStateRequest>();

                    // handle event based on message type
                    Log("Check History Log Validations");
                    var isComplete = IsValidationComplete(provider);

                    if (isComplete)
                    {
                        requests.AddRange(PpmsHelper.MarkBatchDetailAsComplete(provider, _service));

                        // Determine if the provider can be activated
                        var currentStatus = provider.GetAttributeValue<OptionSetValue>("statecode");

                        if (currentStatus != null && currentStatus.Value == (int)AccountState.Active)
                        {
                            //If we get here we have the Provider and Validations are complete. 
                            Log("Set Provider to Active");
                            //Set Provider State/Status
                            requests.Add(PpmsHelper.GetStateRequest(providerRef, currentStatus.Value, (int)Account_StatusCode.Active));
                        }

                        // Execute requests
                        if (requests.Count > 0)
                        {
                            foreach (var request in requests)
                            {
                                _service.Execute(request);
                            }
                        }

                        return;
                    }
                    Log("Validations Not Complete");
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    Log("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    Log("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            Log("Done");
        }

        private Account GetProvider(IOrganizationService service, Guid providerId)
        {
            using (var context = new PpmsContext(service))
            {
                var account = context.AccountSet.FirstOrDefault(a => a.AccountId == providerId);
                context.LoadProperty(account, new Relationship("ppms_account_batchdetail_provider"));
                return account;
            }
        }

        private bool IsValidationComplete(Entity provider)
        {
            using (var svc = new PpmsContext(_service))
            {
                //Retrieve the Validation History Log types
                var validationHistoryLogs = svc.ppms_historylogSet
                            .Where(hl => hl.ppms_providerid.Id == provider.Id && (
                                         hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Contact ||
                                         hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Leie ||
                                         hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.NppesNpi ||
                                         hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Sam ||
                                         hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.CareSite)) 
                            .OrderByDescending(h => h.CreatedOn);

                //Add to list and check to make sure all the needed Validation Types are present
                //check the validationHistoryLogs for null / count
                if (validationHistoryLogs != null) {
                    // Set start date for history log records
                    var startDate = provider.GetAttributeValue<DateTime?>("ppms_validationresetdate");
                    if (!startDate.HasValue)
                        startDate = new DateTime(2000, 1, 1); // Set default minimum date

                    startDate = startDate.Value.ToUniversalTime();

                    var validationHistoryLogsList = validationHistoryLogs.Where(v => v.CreatedOn > startDate).ToList();
                    var recordSource = provider.GetAttributeValue<OptionSetValue>("ppms_recordsource");
                    var providerType = provider.GetAttributeValue<OptionSetValue>("ppms_providertype");

                    // If record source not set, treat as user input (Provider Agreement)
                    if (recordSource == null) recordSource = new OptionSetValue((int)Account_ppms_RecordSource.ProviderAgreement);

                    Log("Source: {0}", recordSource.Value);
                    switch (recordSource.Value)
                    {
                        case (int)Account_ppms_RecordSource.VAProvider:
                            if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Leie)) { return false; }
                            if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.NppesNpi)) { return false; }
                            break;
                        default:
                            if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Contact)) { return false; }
                            if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.Leie)) { return false; }
                            if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.NppesNpi)) { return false; }
                            break;
                    }

                    Log($"Checking for Type 2 provider: {providerType.Value}");
                    if (providerType.Value == (int)ppms_ProviderType.GroupPracticeAgency)
                    {
                        if (!validationHistoryLogsList.Exists(hl => hl.ppms_type.Value == (int)PpmsHelper.HistoryLogType.CareSite)) { return false; }
                    }

                    //If all validations are present return true. 
                    return true; 
                }
                //If no validation logs are found return false. 
                return false;
            }            
        }

        private void Log(string message, params object[] args)
        {
            _tracingService.Trace(message, args);
        }
    }
}
